﻿

//
// extrude.js
//

//
// distance and repeat tool properties
//
var extrudeDistance = 0;
var extrudeRepeat = 1;

//
// we store the tool properties on our command data element
//
var commandData = services.commands.getCommandData("Extrude");
var toolProps;

var hadDistance = commandData.hasTrait("ExtrudeDistance");
var distanceTrait = commandData.getOrCreateTrait("ExtrudeDistance", "float", 0);
if (!hadDistance) {
    // we need to set a default for distance
    distanceTrait.value = 1;
}

//
// special handling for first time access.. we init with default values
//
var hadRepeat = commandData.hasTrait("ExtrudeRepeat");
var repeatTrait = commandData.getOrCreateTrait("ExtrudeRepeat", "float", 0);
if (!hadRepeat) {
    // we didn't have the repeat trait, so set to default value of 1
    repeatTrait.value = 1;
}

//
// store the values from traits into locals
//
extrudeDistance = distanceTrait.value;
extrudeRepeat = repeatTrait.value;

// set the tool props
document.toolProps = commandData;
document.refreshToolPropertyWindow();

function extrude(distance, repeat, geom, polyIndex) {

    var polygonPointCount = geom.getPolygonPointCount(polyIndex);
    if (polygonPointCount < 3) {
        return;
    }

    if (repeat == 0) {
        return;
    }

    var extrudeDir = geom.computePolygonNormal(polyIndex);

    var materialIndex = geom.getPolygonMaterialIndex(polyIndex);

    // this is our starting index for new points
    var newPointStart = geom.pointCount;
    var currentPoint = newPointStart;

    // we gather all the point indices for use when creating polygons
    var pointIndices = new Array();
    for (var i = 0; i < polygonPointCount; i++) {
        var idx = geom.getPolygonPoint(polyIndex, i);
        pointIndices.push(idx);
    }

    // first duplicate the points in the geometry (without adding polygon)
    for (var j = 0; j < repeat; j++) {
        for (var i = 0; i < polygonPointCount; i++) {

            var point = geom.getPointOnPolygon(polyIndex, i);
            point[0] += distance * (j+1) * extrudeDir[0];
            point[1] += distance * (j+1) * extrudeDir[1];
            point[2] += distance * (j+1) * extrudeDir[2];

            geom.addPoints(point, 1);

            pointIndices.push(currentPoint);
            currentPoint++;
        }
    }

    // now create a new polygon
    var newPolyIndex = geom.polygonCount;
    geom.addPolygon(materialIndex);

    var IndexingModePerPointOnPolygon = 3;
    var addTextureCoords = false;

    if (geom.textureCoordinateIndexingMode == IndexingModePerPointOnPolygon)
    {
        addTextureCoords = true;
    }

    // add points to new polygon.
    var newPolyPointStart = newPointStart + (repeat - 1) * polygonPointCount;
    for (var i = 0; i < polygonPointCount; i++) {
        geom.addPolygonPoint(newPolyIndex, newPolyPointStart + i);

        if (addTextureCoords) {
            var texCoord = geom.getTextureCoordinateOnPolygon(polyIndex, i);
            geom.addTextureCoordinates(texCoord, 1);
        }
    }

    // for each segement (we repeat)
    for (var j = 0; j < repeat; j++) {

        // for each edge
        for (var i = 0; i < polygonPointCount; i++) {

            var p0 = i;
            var p1 = i + 1;
            if (p1 >= polygonPointCount) {
                p1 = 0;
            }

            // points i and i + 1 form an edge

            // add a poly containing this edge, edges from each point to the
            // newly duplicted points, and the new edge between the 2 associated duped points
            var thisPolyIndex = geom.polygonCount;
            geom.addPolygon(materialIndex);

            var poly0PointStart = j * polygonPointCount;
            var poly1PointStart = (j + 1) * polygonPointCount;

            var i0 = pointIndices[poly0PointStart + p0];
            var i1 = pointIndices[poly0PointStart + p1];
            var i2 = pointIndices[poly1PointStart + p1];
            var i3 = pointIndices[poly1PointStart + p0];

            geom.addPolygonPoint(thisPolyIndex, i0);
            geom.addPolygonPoint(thisPolyIndex, i1);
            geom.addPolygonPoint(thisPolyIndex, i2);
            geom.addPolygonPoint(thisPolyIndex, i3);

            if (addTextureCoords) {
                var texCoord0 = [0, 0, 1, 0, 1, 1, 0, 1];
                geom.addTextureCoordinates(texCoord0, 4);
            }
        }
    }

    return newPolyIndex;
}

///////////////////////////////////////////////////////////////////////////////
//
// helper to get a designer property as a bool
//
///////////////////////////////////////////////////////////////////////////////
function getDesignerPropAsBool(tname) {
    if (document.designerProps.hasTrait(tname))
        return document.designerProps.getTrait(tname).value;

    return false;
}

function getSelectionMode() {
    if (getDesignerPropAsBool("usePivot"))
        return 0; // default to object mode when using pivot
    if (document.designerProps.hasTrait("SelectionMode"))
        return document.designerProps.getTrait("SelectionMode").value;
    return 0;
}


// find the mesh child
function findFirstChildMeshElement(parent)
{
    // find the mesh child
    for (var i = 0; i < parent.childCount; i++) {

        // get child and its materials
        var child = parent.getChild(i);
        if (child.typeId == "Microsoft.VisualStudio.3D.Mesh") {
            return child;
        }
    }
    return null;
}

function UndoableItem(dist, repeat, collElem, meshElem) {
    this._extrudeDistance = dist;
    this._extrudeRepeat = repeat;

    var clonedColl = collElem.clone();
    this._collectionElem = clonedColl;
    this._polyCollection = clonedColl.behavior;
    this._meshElem = meshElem;
    this._mesh = meshElem.behavior;

    var geom = this._meshElem.getTrait("Geometry").value;
    this._restoreGeom = geom.clone();

    this.getName = function () {
        var IDS_MreUndoExtrude = 146;
        return services.strings.getStringFromId(IDS_MreUndoExtrude);
    }

    this.onDo = function () {

        var geom = this._meshElem.getTrait("Geometry").value;

        var polysToDelete = new Array();
        var polysToSelect = new Array();

        // extrude
        var polyCount = this._polyCollection.getPolygonCount();
        for (var i = 0; i < polyCount; i++) {
            var polyIndex = this._polyCollection.getPolygon(i);
            var newPoly = extrude(this._extrudeDistance, this._extrudeRepeat, geom, polyIndex);

            // services.debug.trace("[Extrude] extruding poly index " + polyIndex);
            // services.debug.trace("[Extrude] adding poly index " + newPoly);

            polysToSelect.push(newPoly);
            polysToDelete.push(polyIndex);
        }

        function sortNumberDescending(a, b) {
            return b - a;
        }

        // delete the old selection
        polysToDelete.sort(sortNumberDescending);

        var numDeletedPolys = polysToDelete.length;

        for (var p = 0; p < polysToDelete.length; p++) {

            // services.debug.trace("[Extrude] removing poly index " + polyIndex);

            geom.removePolygon(polysToDelete[p]);
        }

        var newCollection = this._collectionElem.clone();
        var newPolyCollBeh = newCollection.behavior;

        newPolyCollBeh.clear();

        // shift polygon indices
        for (var p = 0; p < polysToSelect.length; p++) {
            var indexToSelect = polysToSelect[p] - numDeletedPolys;

            // services.debug.trace("[Extrude] selecting poly index " + indexToSelect);

            newPolyCollBeh.addPolygon(indexToSelect);
        }

        this._mesh.selectedObjects = newCollection;
        this._mesh.recomputeCachedGeometry();
    }

    this.onUndo = function () {
        var geom = this._meshElem.getTrait("Geometry").value;
        geom.copyFrom(this._restoreGeom);

        this._mesh.selectedObjects = this._collectionElem;

        this._mesh.recomputeCachedGeometry();
    }
}


if (extrudeRepeat != 0) {

    var selectedElement = document.selectedElement;
    var selectionMode = getSelectionMode();

    // get the poly collection
    var polyCollection = null;
    var mesh = null;
    var meshElem = null;
    var collElem = null;
    if (selectedElement != null) {
        if (selectionMode == 1) {
            meshElem = findFirstChildMeshElement(selectedElement);
            if (meshElem != null) {
                mesh = meshElem.behavior;

                // polygon selection mode
                collElem = mesh.selectedObjects;
                if (collElem != null) {
                    polyCollection = collElem.behavior;
                }
            }
        }
    }

    if (polyCollection != null && collElem.typeId == "PolygonCollection") {
        var undoableItem = new UndoableItem(extrudeDistance, extrudeRepeat, collElem, meshElem);
        undoableItem.onDo();
        services.undoService.addUndoableItem(undoableItem);
    }
}

// SIG // Begin signature block
// SIG // MIIj+QYJKoZIhvcNAQcCoIIj6jCCI+YCAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // kz9XY8lqiFVbcL5uZ5PFVCKKJ4t4jMlzh48n5M8YHxqg
// SIG // gg2TMIIGETCCA/mgAwIBAgITMwAAAI6HkaRXGl/KPgAA
// SIG // AAAAjjANBgkqhkiG9w0BAQsFADB+MQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
// SIG // UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBv
// SIG // cmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQgQ29kZSBT
// SIG // aWduaW5nIFBDQSAyMDExMB4XDTE2MTExNzIyMDkyMVoX
// SIG // DTE4MDIxNzIyMDkyMVowgYMxCzAJBgNVBAYTAlVTMRMw
// SIG // EQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
// SIG // b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
// SIG // b24xDTALBgNVBAsTBE1PUFIxHjAcBgNVBAMTFU1pY3Jv
// SIG // c29mdCBDb3Jwb3JhdGlvbjCCASIwDQYJKoZIhvcNAQEB
// SIG // BQADggEPADCCAQoCggEBANCH1EIrfp3ZxnrUosPjFZLS
// SIG // U52VF8lSNvpUv6sQr+nJ58wmU8PCc79t1gDlANzpamc0
// SIG // MPYWF7QBpZV8i7mkLOaLO3n2Iwx5j/NS30ABHMLGA53r
// SIG // Wc9z6dhxOZvwziVZLdLJWwrvftYyDl10EgTsngRTpmsC
// SIG // Z/hNWYt34Csh4O/ApEUSzwN7A8Y5w9Qi3FVcd0L/nLLl
// SIG // VWdoui12an9mU0fVRwrMON6Ne5cZfYLQJviljuWh8F5k
// SIG // EOT56yfG8uAI0A3yZ8DY8i/7idoV+a4PPgCXB9ELPnDU
// SIG // d6tyeEGYB7gXzKKxX+y981Bno9eU8NKLVY9TppWT5rJm
// SIG // z8k3aORjx88CAwEAAaOCAYAwggF8MB8GA1UdJQQYMBYG
// SIG // CisGAQQBgjdMCAEGCCsGAQUFBwMDMB0GA1UdDgQWBBSr
// SIG // yNbtshXSqo7xzO1sOPdFStCKuzBSBgNVHREESzBJpEcw
// SIG // RTENMAsGA1UECxMETU9QUjE0MDIGA1UEBRMrMjMwMDEy
// SIG // K2IwNTBjNmU3LTc2NDEtNDQxZi1iYzRhLTQzNDgxZTQx
// SIG // NWQwODAfBgNVHSMEGDAWgBRIbmTlUAXTgqoXNzcitW2o
// SIG // ynUClTBUBgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vd3d3
// SIG // Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NybC9NaWNDb2RT
// SIG // aWdQQ0EyMDExXzIwMTEtMDctMDguY3JsMGEGCCsGAQUF
// SIG // BwEBBFUwUzBRBggrBgEFBQcwAoZFaHR0cDovL3d3dy5t
// SIG // aWNyb3NvZnQuY29tL3BraW9wcy9jZXJ0cy9NaWNDb2RT
// SIG // aWdQQ0EyMDExXzIwMTEtMDctMDguY3J0MAwGA1UdEwEB
// SIG // /wQCMAAwDQYJKoZIhvcNAQELBQADggIBAESJAqxpU/PE
// SIG // trvUjGBT58psqElpZr6lmkGZOtid0lcCUWr6v5uW26Ym
// SIG // fQlW6NztJXV6pUdSqB5LFlPz7g+awwSVKcGChKRWMfyg
// SIG // ipGVtb9azqkBH2RGoebK8dd0e7+SCFFefDMCXlE7m+XY
// SIG // Ll8CTAmcGkPace3k2eei2nQsF63lDLUY9VQJ1L4cc80g
// SIG // e6T6yNvY2zqu+pDFo72VZa5GLVcpWNaS8GzaY/GPM6J+
// SIG // OHZe3fM17ayaO2KB0E4ZfEh8sAuPOMwtvNU5ZamVwQPi
// SIG // ksm5q9JXCqrcUgsuViej4piXV468qVluJJKOguIJc4LZ
// SIG // NYPMn3/RBI6IuOKag1iw1JrmMfqUR459puJOefPY02oz
// SIG // FlBw8UK7mAnp/8yVVVsIv5JSqAjE8ejx/0DX+Zo2nf26
// SIG // kIXSVT5QrUYf7yUMuJ46SARj73iYol0DDQLY3CCr5la1
// SIG // 3u8WZsPXVYIeT4J4yZ5UGhBgtxerQBORrrAZwZozne4y
// SIG // cs1lzE9GmC0PUWAefPv+2+gHeQf3oTM4/gma2497tjq9
// SIG // hYa4zLx9ATC3ex2pXRu9zE0X925HM9VA32rKLlG4tbnP
// SIG // wwTTO+Xj6RCM66e63qQuM2opLxRK6h7BIjg1BYXvwgQA
// SIG // DWvB2JYUSBWvflKwuGDEUrVKgreFKgBJKiaDJ1pB3r3V
// SIG // Zkm8C5x4cAm8MIIHejCCBWKgAwIBAgIKYQ6Q0gAAAAAA
// SIG // AzANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UEBhMCVVMx
// SIG // EzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1Jl
// SIG // ZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3Jh
// SIG // dGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJvb3QgQ2Vy
// SIG // dGlmaWNhdGUgQXV0aG9yaXR5IDIwMTEwHhcNMTEwNzA4
// SIG // MjA1OTA5WhcNMjYwNzA4MjEwOTA5WjB+MQswCQYDVQQG
// SIG // EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UE
// SIG // BxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENv
// SIG // cnBvcmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQgQ29k
// SIG // ZSBTaWduaW5nIFBDQSAyMDExMIICIjANBgkqhkiG9w0B
// SIG // AQEFAAOCAg8AMIICCgKCAgEAq/D6chAcLq3YbqqCEE00
// SIG // uvK2WCGfQhsqa+laUKq4BjgaBEm6f8MMHt03a8YS2Avw
// SIG // OMKZBrDIOdUBFDFC04kNeWSHfpRgJGyvnkmc6Whe0t+b
// SIG // U7IKLMOv2akrrnoJr9eWWcpgGgXpZnboMlImEi/nqwhQ
// SIG // z7NEt13YxC4Ddato88tt8zpcoRb0RrrgOGSsbmQ1eKag
// SIG // Yw8t00CT+OPeBw3VXHmlSSnnDb6gE3e+lD3v++MrWhAf
// SIG // TVYoonpy4BI6t0le2O3tQ5GD2Xuye4Yb2T6xjF3oiU+E
// SIG // GvKhL1nkkDstrjNYxbc+/jLTswM9sbKvkjh+0p2ALPVO
// SIG // VpEhNSXDOW5kf1O6nA+tGSOEy/S6A4aN91/w0FK/jJSH
// SIG // vMAhdCVfGCi2zCcoOCWYOUo2z3yxkq4cI6epZuxhH2rh
// SIG // KEmdX4jiJV3TIUs+UsS1Vz8kA/DRelsv1SPjcF0PUUZ3
// SIG // s/gA4bysAoJf28AVs70b1FVL5zmhD+kjSbwYuER8ReTB
// SIG // w3J64HLnJN+/RpnF78IcV9uDjexNSTCnq47f7Fufr/zd
// SIG // sGbiwZeBe+3W7UvnSSmnEyimp31ngOaKYnhfsi+E11ec
// SIG // XL93KCjx7W3DKI8sj0A3T8HhhUSJxAlMxdSlQy90lfdu
// SIG // +HggWCwTXWCVmj5PM4TasIgX3p5O9JawvEagbJjS4NaI
// SIG // jAsCAwEAAaOCAe0wggHpMBAGCSsGAQQBgjcVAQQDAgEA
// SIG // MB0GA1UdDgQWBBRIbmTlUAXTgqoXNzcitW2oynUClTAZ
// SIG // BgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8E
// SIG // BAMCAYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAW
// SIG // gBRyLToCMZBDuRQFTuHqp8cx0SOJNDBaBgNVHR8EUzBR
// SIG // ME+gTaBLhklodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20v
// SIG // cGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXQyMDEx
// SIG // XzIwMTFfMDNfMjIuY3JsMF4GCCsGAQUFBwEBBFIwUDBO
// SIG // BggrBgEFBQcwAoZCaHR0cDovL3d3dy5taWNyb3NvZnQu
// SIG // Y29tL3BraS9jZXJ0cy9NaWNSb29DZXJBdXQyMDExXzIw
// SIG // MTFfMDNfMjIuY3J0MIGfBgNVHSAEgZcwgZQwgZEGCSsG
// SIG // AQQBgjcuAzCBgzA/BggrBgEFBQcCARYzaHR0cDovL3d3
// SIG // dy5taWNyb3NvZnQuY29tL3BraW9wcy9kb2NzL3ByaW1h
// SIG // cnljcHMuaHRtMEAGCCsGAQUFBwICMDQeMiAdAEwAZQBn
// SIG // AGEAbABfAHAAbwBsAGkAYwB5AF8AcwB0AGEAdABlAG0A
// SIG // ZQBuAHQALiAdMA0GCSqGSIb3DQEBCwUAA4ICAQBn8oal
// SIG // mOBUeRou09h0ZyKbC5YR4WOSmUKWfdJ5DJDBZV8uLD74
// SIG // w3LRbYP+vj/oCso7v0epo/Np22O/IjWll11lhJB9i0ZQ
// SIG // VdgMknzSGksc8zxCi1LQsP1r4z4HLimb5j0bpdS1HXeU
// SIG // OeLpZMlEPXh6I/MTfaaQdION9MsmAkYqwooQu6SpBQyb
// SIG // 7Wj6aC6VoCo/KmtYSWMfCWluWpiW5IP0wI/zRive/DvQ
// SIG // vTXvbiWu5a8n7dDd8w6vmSiXmE0OPQvyCInWH8MyGOLw
// SIG // xS3OW560STkKxgrCxq2u5bLZ2xWIUUVYODJxJxp/sfQn
// SIG // +N4sOiBpmLJZiWhub6e3dMNABQamASooPoI/E01mC8Cz
// SIG // TfXhj38cbxV9Rad25UAqZaPDXVJihsMdYzaXht/a8/jy
// SIG // FqGaJ+HNpZfQ7l1jQeNbB5yHPgZ3BtEGsXUfFL5hYbXw
// SIG // 3MYbBL7fQccOKO7eZS/sl/ahXJbYANahRr1Z85elCUtI
// SIG // EJmAH9AAKcWxm6U/RXceNcbSoqKfenoi+kiVH6v7RyOA
// SIG // 9Z74v2u3S5fi63V4GuzqN5l5GEv/1rMjaHXmr/r8i+sL
// SIG // gOppO6/8MO0ETI7f33VtY5E90Z1WTk+/gFcioXgRMiF6
// SIG // 70EKsT/7qMykXcGhiJtXcVZOSEXAQsmbdlsKgEhr/Xmf
// SIG // wb1tbWrJUnMTDXpQzTGCFb4wghW6AgEBMIGVMH4xCzAJ
// SIG // BgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAw
// SIG // DgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3Nv
// SIG // ZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29m
// SIG // dCBDb2RlIFNpZ25pbmcgUENBIDIwMTECEzMAAACOh5Gk
// SIG // Vxpfyj4AAAAAAI4wDQYJYIZIAWUDBAIBBQCggawwGQYJ
// SIG // KoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGC
// SIG // NwIBCzEOMAwGCisGAQQBgjcCARUwLwYJKoZIhvcNAQkE
// SIG // MSIEIFlMSDuN0Oz9t1zUJ6+B4mpPPv+IHnEUeS30PUFy
// SIG // mk6aMEAGCisGAQQBgjcCAQwxMjAwoBaAFABFAHgAdABy
// SIG // AHUAZABlAC4AagBzoRaAFGh0dHA6Ly9taWNyb3NvZnQu
// SIG // Y29tMA0GCSqGSIb3DQEBAQUABIIBAAcaMr6nmgU0Tv+/
// SIG // KUoyDaX8/GKsLfA+80kTMxGTiHYIAiisSHJn1S42/eYV
// SIG // vdFEt3KrIqIWfRC1DjIecSZ81e4Y5ENpMfo051I8kTOF
// SIG // w+lQeewOB8tDczQXGTnHF8pLwhMdBZQ9DiwAi9YMBJRQ
// SIG // 6kncf2OH0+XYHUUaNzVIAK7icucLkC4yHzn/8MMWQ7Ol
// SIG // ghJanitzNutiVOfMoJiG0NaUWNj3HR60cz5/OwlggP9b
// SIG // c1JN09YvSGHytmjXAttUT5g/qV3A8ZPS0WGINxGxeDol
// SIG // vaSOGUAJgAqZ2/11ElFtV6I/CSAcSorUoeBXVqZwg19b
// SIG // ETHqwWfPh1RH8nL8bIShghNKMIITRgYKKwYBBAGCNwMD
// SIG // ATGCEzYwghMyBgkqhkiG9w0BBwKgghMjMIITHwIBAzEP
// SIG // MA0GCWCGSAFlAwQCAQUAMIIBPQYLKoZIhvcNAQkQAQSg
// SIG // ggEsBIIBKDCCASQCAQEGCisGAQQBhFkKAwEwMTANBglg
// SIG // hkgBZQMEAgEFAAQgMdLqdK5Ugsi+IvSIA2gC+UCMxqmG
// SIG // ktTfKCwSjhQlK9UCBlivgefcNBgTMjAxNzAzMDQyMzI0
// SIG // MDYuNDI0WjAHAgEBgAIB9KCBuaSBtjCBszELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjENMAsGA1UECxMETU9QUjEnMCUGA1UE
// SIG // CxMebkNpcGhlciBEU0UgRVNOOkIxQjctRjY3Ri1GRUMy
// SIG // MSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFtcCBT
// SIG // ZXJ2aWNloIIOzTCCBnEwggRZoAMCAQICCmEJgSoAAAAA
// SIG // AAIwDQYJKoZIhvcNAQELBQAwgYgxCzAJBgNVBAYTAlVT
// SIG // MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
// SIG // ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9y
// SIG // YXRpb24xMjAwBgNVBAMTKU1pY3Jvc29mdCBSb290IENl
// SIG // cnRpZmljYXRlIEF1dGhvcml0eSAyMDEwMB4XDTEwMDcw
// SIG // MTIxMzY1NVoXDTI1MDcwMTIxNDY1NVowfDELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRp
// SIG // bWUtU3RhbXAgUENBIDIwMTAwggEiMA0GCSqGSIb3DQEB
// SIG // AQUAA4IBDwAwggEKAoIBAQCpHQ28dxGKOiDs/BOX9fp/
// SIG // aZRrdFQQ1aUKAIKF++18aEssX8XD5WHCdrc+Zitb8BVT
// SIG // JwQxH0EbGpUdzgkTjnxhMFmxMEQP8WCIhFRDDNdNuDgI
// SIG // s0Ldk6zWczBXJoKjRQ3Q6vVHgc2/JGAyWGBG8lhHhjKE
// SIG // HnRhZ5FfgVSxz5NMksHEpl3RYRNuKMYa+YaAu99h/EbB
// SIG // Jx0kZxJyGiGKr0tkiVBisV39dx898Fd1rL2KQk1AUdEP
// SIG // nAY+Z3/1ZsADlkR+79BL/W7lmsqxqPJ6Kgox8NpOBpG2
// SIG // iAg16HgcsOmZzTznL0S6p/TcZL2kAcEgCZN4zfy8wMlE
// SIG // XV4WnAEFTyJNAgMBAAGjggHmMIIB4jAQBgkrBgEEAYI3
// SIG // FQEEAwIBADAdBgNVHQ4EFgQU1WM6XIoxkPNDe3xGG8Uz
// SIG // aFqFbVUwGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEw
// SIG // CwYDVR0PBAQDAgGGMA8GA1UdEwEB/wQFMAMBAf8wHwYD
// SIG // VR0jBBgwFoAU1fZWy4/oolxiaNE9lJBb186aGMQwVgYD
// SIG // VR0fBE8wTTBLoEmgR4ZFaHR0cDovL2NybC5taWNyb3Nv
// SIG // ZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvTWljUm9vQ2Vy
// SIG // QXV0XzIwMTAtMDYtMjMuY3JsMFoGCCsGAQUFBwEBBE4w
// SIG // TDBKBggrBgEFBQcwAoY+aHR0cDovL3d3dy5taWNyb3Nv
// SIG // ZnQuY29tL3BraS9jZXJ0cy9NaWNSb29DZXJBdXRfMjAx
// SIG // MC0wNi0yMy5jcnQwgaAGA1UdIAEB/wSBlTCBkjCBjwYJ
// SIG // KwYBBAGCNy4DMIGBMD0GCCsGAQUFBwIBFjFodHRwOi8v
// SIG // d3d3Lm1pY3Jvc29mdC5jb20vUEtJL2RvY3MvQ1BTL2Rl
// SIG // ZmF1bHQuaHRtMEAGCCsGAQUFBwICMDQeMiAdAEwAZQBn
// SIG // AGEAbABfAFAAbwBsAGkAYwB5AF8AUwB0AGEAdABlAG0A
// SIG // ZQBuAHQALiAdMA0GCSqGSIb3DQEBCwUAA4ICAQAH5ohR
// SIG // DeLG4Jg/gXEDPZ2joSFvs+umzPUxvs8F4qn++ldtGTCz
// SIG // wsVmyWrf9efweL3HqJ4l4/m87WtUVwgrUYJEEvu5U4zM
// SIG // 9GASinbMQEBBm9xcF/9c+V4XNZgkVkt070IQyK+/f8Z/
// SIG // 8jd9Wj8c8pl5SpFSAK84Dxf1L3mBZdmptWvkx872ynoA
// SIG // b0swRCQiPM/tA6WWj1kpvLb9BOFwnzJKJ/1Vry/+tuWO
// SIG // M7tiX5rbV0Dp8c6ZZpCM/2pif93FSguRJuI57BlKcWOd
// SIG // eyFtw5yjojz6f32WapB4pm3S4Zz5Hfw42JT0xqUKloak
// SIG // vZ4argRCg7i1gJsiOCC1JeVk7Pf0v35jWSUPei45V3ai
// SIG // caoGig+JFrphpxHLmtgOR5qAxdDNp9DvfYPw4TtxCd9d
// SIG // dJgiCGHasFAeb73x4QDf5zEHpJM692VHeOj4qEir995y
// SIG // fmFrb3epgcunCaw5u+zGy9iCtHLNHfS4hQEegPsbiSpU
// SIG // ObJb2sgNVZl6h3M7COaYLeqN4DMuEin1wC9UJyH3yKxO
// SIG // 2ii4sanblrKnQqLJzxlBTeCG+SqaoxFmMNO7dDJL32N7
// SIG // 9ZmKLxvHIa9Zta7cRDyXUHHXodLFVeNp3lfB0d4wwP3M
// SIG // 5k37Db9dT+mdHhk4L7zPWAUu7w2gUDXa7wknHNWzfjUe
// SIG // CLraNtvTX4/edIhJEjCCBNowggPCoAMCAQICEzMAAACx
// SIG // cRN533X2NcgAAAAAALEwDQYJKoZIhvcNAQELBQAwfDEL
// SIG // MAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24x
// SIG // EDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jv
// SIG // c29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9z
// SIG // b2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAwHhcNMTYwOTA3
// SIG // MTc1NjU3WhcNMTgwOTA3MTc1NjU3WjCBszELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjENMAsGA1UECxMETU9QUjEnMCUGA1UE
// SIG // CxMebkNpcGhlciBEU0UgRVNOOkIxQjctRjY3Ri1GRUMy
// SIG // MSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFtcCBT
// SIG // ZXJ2aWNlMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIB
// SIG // CgKCAQEAqqQklG1Y1lu8ob0P7deumuRn4JvRi2GErmK9
// SIG // 4vgbnWPmd0j/9arA7539HD1dpG1uhYbmnAxc+qsuvMM0
// SIG // fpEvttTK4lZSU7ss5rJfWmbFn/J8kSGI8K9iBaB6hQkJ
// SIG // uIX4si9ppNr9R3oZI3HbJ/yRkKUPk4hozpY6CkehRc0/
// SIG // Zfu6tQiyqI7mClXYZTXjw+rLsh3/gdBvYDd38zFBllaf
// SIG // +3uimKQgUTXGjbKfqZZk3tEU3ibWVPUxAmmxlG3sWTlX
// SIG // mU31fCw/6TVzGg251lq+Q46OjbeH9vB2TOcqEso4Nai3
// SIG // J1CdMAYUdlelVVtgQdIx/c+5Hvrw0Y6W7uGBAWnW5wID
// SIG // AQABo4IBGzCCARcwHQYDVR0OBBYEFE5XPfeLLhRLV7L8
// SIG // Il7Tz7cnRBA7MB8GA1UdIwQYMBaAFNVjOlyKMZDzQ3t8
// SIG // RhvFM2hahW1VMFYGA1UdHwRPME0wS6BJoEeGRWh0dHA6
// SIG // Ly9jcmwubWljcm9zb2Z0LmNvbS9wa2kvY3JsL3Byb2R1
// SIG // Y3RzL01pY1RpbVN0YVBDQV8yMDEwLTA3LTAxLmNybDBa
// SIG // BggrBgEFBQcBAQROMEwwSgYIKwYBBQUHMAKGPmh0dHA6
// SIG // Ly93d3cubWljcm9zb2Z0LmNvbS9wa2kvY2VydHMvTWlj
// SIG // VGltU3RhUENBXzIwMTAtMDctMDEuY3J0MAwGA1UdEwEB
// SIG // /wQCMAAwEwYDVR0lBAwwCgYIKwYBBQUHAwgwDQYJKoZI
// SIG // hvcNAQELBQADggEBAHPujfu0W8PBTpjfYaPrAKIBLKcl
// SIG // jT4+YnWbbgGvmXU8OvIUDBkkv8gNGGHRO5DSySaCARIz
// SIG // gn2yIheAqh6GwM2yKrfb4eVCYPe1CTlCseS5TOv+Tn/9
// SIG // 5mXj+NxTqvuNmrhgCVr0CQ7b3xoKcwDcQbg7TmerDgbI
// SIG // v2k7cEqbYbU/B3MtSX8Zjjf0ZngdKoX0JYkAEDbZchOr
// SIG // RiUtDJItegPKZPf6CjeHYjrmKwvTOVCzv3lW0uyh1yb/
// SIG // ODeRH+VqENSHCboFiEiq9KpKMOpek1VvQhmI2KbTlRvK
// SIG // 869gj1NwuUHH8c3WXu4A0X1+CBmU8t0gvd/fFlQvw04v
// SIG // eKWh986hggN2MIICXgIBATCB46GBuaSBtjCBszELMAkG
// SIG // A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAO
// SIG // BgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29m
// SIG // dCBDb3Jwb3JhdGlvbjENMAsGA1UECxMETU9QUjEnMCUG
// SIG // A1UECxMebkNpcGhlciBEU0UgRVNOOkIxQjctRjY3Ri1G
// SIG // RUMyMSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFt
// SIG // cCBTZXJ2aWNloiUKAQEwCQYFKw4DAhoFAAMVADq635Mo
// SIG // ZeR60+ej9uKnRG5YqlPSoIHCMIG/pIG8MIG5MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMQ0wCwYDVQQLEwRNT1BSMScwJQYD
// SIG // VQQLEx5uQ2lwaGVyIE5UUyBFU046NERFOS0wQzVFLTNF
// SIG // MDkxKzApBgNVBAMTIk1pY3Jvc29mdCBUaW1lIFNvdXJj
// SIG // ZSBNYXN0ZXIgQ2xvY2swDQYJKoZIhvcNAQEFBQACBQDc
// SIG // ZXBzMCIYDzIwMTcwMzA0MTY1NzIzWhgPMjAxNzAzMDUx
// SIG // NjU3MjNaMHQwOgYKKwYBBAGEWQoEATEsMCowCgIFANxl
// SIG // cHMCAQAwBwIBAAICIs0wBwIBAAICGYEwCgIFANxmwfMC
// SIG // AQAwNgYKKwYBBAGEWQoEAjEoMCYwDAYKKwYBBAGEWQoD
// SIG // AaAKMAgCAQACAxbjYKEKMAgCAQACAwehIDANBgkqhkiG
// SIG // 9w0BAQUFAAOCAQEACsgyG3dK0zXo8u/zrdHnbUvsxOhP
// SIG // pFGQbKIDgCAneTBWSRf+IuyxpmAWpSEtH4ir8t79lvKq
// SIG // T3AP3Jti81chBUiQgdldj+BG6vG/KVU7J4qwTBto4Nmw
// SIG // heazLc/Fmnz5Yufu9fViTaT1/P5DnycDm/LBPlS+MDad
// SIG // C1llG6EFQu8M597RC6Tnnhlj6TjWPWSSdpoptcB6my1z
// SIG // SdJLEoSGlrFO1EK5PsxsaUsldC7YjpdENpGu8v0T/2/e
// SIG // 5mvYsMHPXJVHk/osW8zDi3zcErAPGJi0uZdyK8iVWTRq
// SIG // ru37jaPK1MR/aaBcsTc+jMnzZ8QwJxyVRCGIZD0iId30
// SIG // pERe8DGCAvUwggLxAgEBMIGTMHwxCzAJBgNVBAYTAlVT
// SIG // MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
// SIG // ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9y
// SIG // YXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0
// SIG // YW1wIFBDQSAyMDEwAhMzAAAAsXETed919jXIAAAAAACx
// SIG // MA0GCWCGSAFlAwQCAQUAoIIBMjAaBgkqhkiG9w0BCQMx
// SIG // DQYLKoZIhvcNAQkQAQQwLwYJKoZIhvcNAQkEMSIEIGIV
// SIG // D5avfZK4YMf8PozPvkpev5ZArmG9ZADGeWcYyVT7MIHi
// SIG // BgsqhkiG9w0BCRACDDGB0jCBzzCBzDCBsQQUOrrfkyhl
// SIG // 5HrT56P24qdEbliqU9IwgZgwgYCkfjB8MQswCQYDVQQG
// SIG // EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UE
// SIG // BxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENv
// SIG // cnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGlt
// SIG // ZS1TdGFtcCBQQ0EgMjAxMAITMwAAALFxE3nfdfY1yAAA
// SIG // AAAAsTAWBBTMp2CKoD2I0KCj48K5lur8G0AEaTANBgkq
// SIG // hkiG9w0BAQsFAASCAQBMb+IU8SLwrcp2ZKcoRLABEpPs
// SIG // y8aT60P/y33geFwgB1QIvDtS8qyziQlo/WSZ5jw/FMnt
// SIG // hy9UqibYwHObVW16iT+u3vGvV9nvpdd2PYk3ITkQMQHE
// SIG // i8e6aX8JtteeghFfhQ42SA3jjh/5b8QHDGjhdGOB9uV2
// SIG // cFbBF7KoSLIeJAH7qMuKewj/6idXfS4+jmJHcglwLydr
// SIG // dlSza6IcI2wM8FqNYvn3tJsTTRXeYWzbqrjwO2dwLjeH
// SIG // tgV+p5tp0meZnvJfgtzj0uGqRM7QOtfeK4xBC2JeUBXh
// SIG // 3R6rYuXr1Z5j8Wi7Uso7mYpCal/9E1ASAb9J01PQyImO
// SIG // GPUlML0U
// SIG // End signature block
